"use client";

import { useEffect, useState } from "react";
import wordpressAPI from "@/lib/wordpress";
import ContentCard from "@/components/cards/ContentCard";
import SectionHeader from "@/components/common/SectionHeader";
import { useLanguage } from "@/contexts/LanguageContext";

export default function PortfolioPage() {
  const { language } = useLanguage();
  const [portfolio, setPortfolio] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);

  // ✅ Control how many portfolio items to load
  const LIMIT = parseInt(process.env.NEXT_PUBLIC_POSTS_LIMIT || "6", 10);

  useEffect(() => {
    const fetchPortfolio = async () => {
      try {
        setLoading(true);
        const data = await wordpressAPI.getPostsByType("portfolio", language, LIMIT);
        setPortfolio(data);
      } catch (error) {
        console.error("Error loading portfolio:", error);
      } finally {
        setLoading(false);
      }
    };

    fetchPortfolio();
  }, [language]);

  return (
    <div className="te-portfolio mt-12">
      <div className="container my-8">
        <SectionHeader
          title={language === "ar" ? "بورتفوليو" : "Portfolio"}
          subtitle={
            language === "ar"
              ? "استعراض نماذج أعمالنا وإبداعاتنا"
              : "Showcasing our work and creativity"
          }
          level="h2"
          textAlign="start"
        />

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
          {loading
            ? Array.from({ length: LIMIT }).map((_, i) => (
              <div
                key={i}
                className="animate-pulse bg-white dark:bg-gray-800 rounded-lg shadow-md overflow-hidden"
              >
                {/* Thumbnail skeleton */}
                <div className="aspect-[3/2] w-full bg-gray-200 dark:bg-gray-700" />

                {/* Text skeleton */}
                <div className="p-6 space-y-4">
                  <div className="h-6 bg-gray-200 dark:bg-gray-700 rounded w-3/4" />
                  <div className="h-4 bg-gray-200 dark:bg-gray-700 rounded w-full" />
                  <div className="h-4 bg-gray-200 dark:bg-gray-700 rounded w-5/6" />
                </div>
              </div>
            ))
            : portfolio.map((item) => (
              <ContentCard key={item.id} post={item} basePath="portfolio" />
            ))}
        </div>
      </div>
    </div>
  );
}
