"use client";

import { useSearchParams } from "next/navigation";
import { useEffect, useState } from "react";
import wordpressAPI from "@/lib/wordpress";
import { useLanguage } from "@/contexts/LanguageContext";
import Link from "next/link";

export default function SearchPageClient() {
  const searchParams = useSearchParams();
  const query = searchParams.get("query") || "";
  const { language } = useLanguage(); // Ensure 'language' is correctly typed as 'en' | 'ar'

  const [results, setResults] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    const fetchResults = async () => {
      if (!query) {
        setLoading(false);
        return;
      }
      setLoading(true);
      const res = await wordpressAPI.search(query, language);
      setResults(res);
      setLoading(false);
    };
    fetchResults();
  }, [query, language]);

  const formatSearchUrl = (item: any) => {
    let slug = item.url.replace(process.env.NEXT_PUBLIC_WORDPRESS_URL!, "");
    slug = slug.replace(/^\/(ar|en)(\/|$)/, "/"); // ✅ شيل أي prefix لغة
    if (item.subtype === "post") return `/posts${slug}`;
    if (item.subtype === "page") return slug;
    if (item.subtype === "project") return `/project${slug}`;
    return slug;
  };

  // ✅ الترجمة
  const t: {
    en: { title: string; noResults: string };
    ar: { title: string; noResults: string };
  } = {
    en: {
      title: "Search results for:",
      noResults: "No results found.",
    },
    ar: {
      title: "نتائج البحث عن:",
      noResults: "لم يتم العثور على نتائج.",
    },
  };

  return (
    <div className="te-search-results">
      <div className="container my-8">
        <h1 className="text-2xl md:text-3xl font-bold mb-4 text-gray-900 dark:text-white">
          {t[language as keyof typeof t]?.title || t.en.title}{" "}
          <span className="text-primary-600 dark:text-primary-400">{query}</span>
        </h1>

        {/* ✅ Skeleton أثناء التحميل */}
        {loading && (
          <div className="space-y-3 animate-pulse">
            {[...Array(4)].map((_, i) => (
              <div
                key={i}
                className="p-3 border border-gray-200 dark:border-gray-700 rounded bg-gray-100 dark:bg-gray-700 h-16"
              />
            ))}
          </div>
        )}

        {!loading && results.length === 0 && (
          <p className="text-gray-700 dark:text-gray-300">
            {t[language as keyof typeof t]?.noResults || t.en.noResults}
          </p>
        )}

        {!loading && results.length > 0 && (
          <div className="space-y-3">
            {results.map((item) => (
              <div key={item.id}>
                <Link
                  href={formatSearchUrl(item)}
                  className="block p-3 border rounded bg-white dark:bg-gray-800 border-gray-200 dark:border-gray-700 hover:bg-gray-50 dark:hover:bg-gray-700"
                >
                  <h2 className="font-semibold text-gray-900 dark:text-white">
                    {item.title}
                  </h2>
                  <p className="text-sm text-gray-500 dark:text-gray-400">
                    {item.subtype.toUpperCase()}
                  </p>
                </Link>
              </div>
            ))}
          </div>
        )}
      </div>
    </div>
  );
}
