import Link from "next/link";
import Image from "next/image";
import { useLanguage } from "@/contexts/LanguageContext";
import Icon from "@/components/icons/Icon";

interface ContentCardProps {
    post: any;
    basePath: string;
}

export default function ContentCard({ post, basePath }: ContentCardProps) {
    const { language } = useLanguage();

    // ✅ Extract image sizes
    const sizes = post._embedded?.["wp:featuredmedia"]?.[0]?.media_details?.sizes;
    const featuredImage =
        sizes?.["card-thumb"]?.source_url ||
        sizes?.["medium"]?.source_url ||
        post._embedded?.["wp:featuredmedia"]?.[0]?.source_url || null;

    const altText =
        post._embedded?.["wp:featuredmedia"]?.[0]?.alt_text ||
        post.title.rendered;

    // ✅ Extract category (first one)
    const category = post._embedded?.["wp:term"]?.[0]?.[0] || null;

    // ✅ Extract author
    const author = post._embedded?.["author"]?.[0] || null;

    // ✅ Extract date
    const postDate = new Date(post.date);

    // ✅ Helper function to get category URL
    const getCategoryUrl = (category: any, basePath: string) => {
        if (!category) return '#';

        // If category has a slug, use it directly
        if (category.slug) {
            // Dynamically generate the URL based on the basePath (custom post type)
            return `/${basePath}/category/${category.slug}`;
        }

        // If no slug but has link, extract slug from WordPress URL
        if (category.link) {
            try {
                const url = new URL(category.link);
                let slug = url.pathname;
                slug = slug.replace(/^\/|\/$/g, ''); // Remove leading/trailing slashes

                // Handle different WordPress category URL structures
                if (slug.startsWith('category/')) {
                    slug = slug.replace('category/', '');
                } else if (slug.includes('/category/')) {
                    const parts = slug.split('/category/');
                    slug = parts[1] || slug;
                }
                slug = slug.replace(/\//g, ''); // Remove any remaining slashes

                // Dynamically adjust the URL path based on the basePath
                return `/${basePath}/category/${slug}`;
            } catch (error) {
                console.error('Error parsing category URL:', error);
                return '#';
            }
        }

        return '#';
    };

    return (
        <article
            id={`post-${post.id}`}
            className={`post-${post.id} ${post.type} flex flex-col h-full`}
        >
            <div className="te-card te-card-image-classic te-card-hover flex flex-col h-full bg-white dark:bg-gray-800 rounded-xl border border-gray-200 dark:border-gray-700 overflow-hidden relative">

                {/* Image */}
                <div className="te-card-image-container aspect-[16/9] relative">
                    {featuredImage ? (
                        <Link
                            href={`/${basePath}/${post.slug}`}
                            title={altText}
                            className="block w-full h-full"
                        >
                            <Image
                                src={featuredImage}
                                alt={altText}
                                fill
                                sizes="(max-width: 640px) 100vw, (max-width: 1024px) 50vw, 480px"
                                className="te-card-image w-full h-full object-cover transition-transform duration-300 ease-in-out"
                                priority
                                fetchPriority="high"
                            />
                        </Link>
                    ) : (
                        <div className="placeholder bg-gray-200 dark:bg-gray-700 w-full h-full flex items-center justify-center">
                            <span className="text-gray-300 dark:text-gray-600">
                                <Icon name="imagePlus" size={42} />
                            </span>
                        </div>
                    )}

                    {/* Category */}
                    {category && (
                        <div className="te-card-category absolute top-3 start-3 bg-primary-900/80 hover:bg-primary-700 text-white px-3 py-1 block rounded-full text-xs font-medium uppercase tracking-wide z-10">
                            <Link href={getCategoryUrl(category, basePath)}>
                                {category.name}
                            </Link>
                        </div>
                    )}
                </div>

                {/* Body */}
                <div className="te-card-body flex-1 p-5 sm:p-6">
                    <h3 className="te-card-title text-lg font-semibold text-gray-900 dark:text-white mb-2 leading-tight">
                        <Link
                            href={`/${basePath}/${post.slug}`}
                            className="hover:underline"
                            dangerouslySetInnerHTML={{ __html: post.title.rendered }}
                        />
                    </h3>
                    <p
                        className="te-card-text text-sm text-gray-700 dark:text-gray-200 leading-relaxed mt-4 line-clamp-3"
                        dangerouslySetInnerHTML={{ __html: post.excerpt.rendered }}
                    />
                </div>

                {/* Footer */}
                <div className="te-card-footer mt-auto px-5 py-4 sm:px-5 sm:py-4 border-t border-gray-200 dark:border-gray-600 bg-gray-50 dark:bg-gray-900">
                    <div className="te-card-meta flex items-center justify-between text-xs text-gray-500 dark:text-gray-400">

                        {/* Author */}
                        {author && (
                            <div className="te-card-author flex items-center gap-2">
                                <span className="te-card-author-avatar w-10 h-10 rounded-full bg-gradient-to-br from-primary-500 to-secondary-500 flex items-center justify-center text-white font-semibold">
                                    {author.name?.charAt(0) || "?"}
                                </span>
                                {author.name}
                            </div>
                        )}

                        {/* Date */}
                        <div className="te-card-date text-xs text-gray-500 dark:text-gray-400">
                            <time dateTime={post.date}>
                                {postDate.toLocaleDateString(
                                    language === "ar" ? "ar-EG" : "en-US",
                                    { month: "short", day: "numeric", year: "numeric" }
                                )}
                            </time>
                        </div>
                    </div>
                </div>

            </div>
        </article>
    );
}