"use client";

import { useEffect, useState } from "react";
import Icon from "@/components/icons/Icon";

export default function DarkModeToggle() {
    const [isDark, setIsDark] = useState(false);

    // ✅ Check system & localStorage on mount
    useEffect(() => {
        const saved = localStorage.getItem("darkMode");
        const systemDark = window.matchMedia("(prefers-color-scheme: dark)").matches;
        const active = saved === "true" || (!saved && systemDark);

        setIsDark(active);
        document.documentElement.classList.toggle("dark", active);

        // ✅ Listen for system changes
        const media = window.matchMedia("(prefers-color-scheme: dark)");
        const systemListener = (e: MediaQueryListEvent) => {
            if (localStorage.getItem("darkMode") === null) {
                setIsDark(e.matches);
                document.documentElement.classList.toggle("dark", e.matches);
            }
        };
        media.addEventListener("change", systemListener);
        return () => media.removeEventListener("change", systemListener);
    }, []);

    // ✅ Apply theme + save
    const applyTheme = (dark: boolean) => {
        setIsDark(dark);
        document.documentElement.classList.toggle("dark", dark);
        localStorage.setItem("darkMode", dark ? "true" : "false");
    };

    return (
        <button
            onClick={() => applyTheme(!isDark)}
            aria-label={isDark ? "Switch to light mode" : "Switch to dark mode"}
            className="w-10 h-10 flex items-center justify-center text-gray-900 dark:text-white
                        rounded-md 
                        border border-gray-300 dark:border-gray-600 
                        transition-transform duration-300 
                        focus:outline-none focus:ring-0"
        >
            {isDark ? (<Icon name="moon" size={18} />) : (<Icon name="sun" size={20} />)}
        </button>
    );
}
