import axios from 'axios';

// 📌 نجيب رابط الـ API من ملف .env.local
const API_URL = process.env.NEXT_PUBLIC_WORDPRESS_URL;

// 📌 لو مفيش رابط API متعرف → يرمي Error
if (!API_URL) {
    throw new Error('NEXT_PUBLIC_WORDPRESS_URL is not defined');
}

// 📌 تعريف شكل عناصر المينيو
interface MenuItem {
    id: number;
    title: string;
    url: string;
    target?: string;
    parent: string | number;
}

// 📌 تعريف شكل البوستات
interface Post {
    id: number;
    slug: string;
    title: {
        rendered: string;
    };
    content: {
        rendered: string;
    };
    excerpt: {
        rendered: string;
    };
    date: string;
}

// 📌 تعريف شكل الصفحات
interface Page {
    id: number;
    slug: string;
    title: {
        rendered: string;
    };
    content: {
        rendered: string;
    };
}

// 📌 كلاس للتعامل مع API بتاع ووردبريس
class WordPressAPI {
    private apiUrl: string;

    constructor() {
        this.apiUrl = API_URL!;
    }

    // 📌 دالة مساعدة لإضافة اللغة كـ parameter
    private getLanguageParam(lang?: string): string {
        return lang ? `?lang=${lang}` : '';
    }

    // 📌 جلب القوائم المخصصة
    async getMenus(lang?: string): Promise<Record<string, MenuItem[]>> {
        try {
            const response = await axios.get(
                `${this.apiUrl}/wp-json/custom/v1/menus${this.getLanguageParam(lang)}`
            );
            return response.data;
        } catch (error) {
            console.error('Error fetching menus:', error);
            return {
                'header-menu': [],
                'footer-menu-1': [],
                'footer-menu-2': []
            };
        }
    }

    // 📌 البحث في ووردبريس
    async search(query: string, lang?: string) {
        if (!query) return [];
        try {
            const langParam = this.getLanguageParam(lang);
            const separator = langParam ? "&" : "?";
            const response = await axios.get(
                `${this.apiUrl}/wp-json/wp/v2/search${langParam}${separator}search=${encodeURIComponent(query)}&per_page=10`
            );
            return response.data;
        } catch (error) {
            console.error("Error searching:", error);
            return [];
        }
    }

    // 📌 جلب البوستات مع دعم الباجينيشن
    async getPosts(
        lang?: string,
        perPage: number = 10,
        page: number = 1
    ): Promise<{ items: Post[]; totalPages: number }> {
        try {
            const langParam = this.getLanguageParam(lang);
            const separator = langParam ? "&" : "?";

            const response = await axios.get(
                `${this.apiUrl}/wp-json/wp/v2/posts${langParam}${separator}per_page=${perPage}&page=${page}&_embed`
            );

            return {
                items: response.data,
                totalPages: parseInt(response.headers["x-wp-totalpages"] || "1", 10),
            };
        } catch (error) {
            console.error("Error fetching posts:", error);
            return { items: [], totalPages: 1 };
        }
    }

    // 📌 جلب بوست واحد عن طريق الـ slug
    async getPostBySlug(slug: string, lang?: string): Promise<Post | null> {
        try {
            const langParam = this.getLanguageParam(lang);
            const separator = langParam ? '&' : '?';
            let response = await axios.get(
                `${this.apiUrl}/wp-json/wp/v2/posts${langParam}${separator}slug=${slug}&_embed`
            );

            // ✅ fallback لو مفيش ترجمة بالعربي أو الإنجليزي
            if ((!response.data || response.data.length === 0) && lang && lang !== "en") {
                response = await axios.get(
                    `${this.apiUrl}/wp-json/wp/v2/posts?lang=en&slug=${slug}&_embed`
                );
            }

            return response.data[0] || null;
        } catch (error) {
            console.error('Error fetching post:', error);
            return null;
        }
    }

    // 📌 جلب كل الصفحات
    async getPages(lang?: string): Promise<Page[]> {
        try {
            const langParam = this.getLanguageParam(lang);
            const separator = langParam ? "&" : "?";
            const response = await axios.get(
                `${this.apiUrl}/wp-json/wp/v2/pages${langParam}${separator}_embed` // ✅ تأكدنا من &_embed
            );
            return response.data;
        } catch (error) {
            console.error('Error fetching pages:', error);
            return [];
        }
    }

    // 📌 جلب صفحة واحدة عن طريق الـ slug
    async getPageBySlug(slug: string, lang?: string): Promise<Page | null> {
        try {
            const langParam = this.getLanguageParam(lang);
            const separator = langParam ? '&' : '?';
            let response = await axios.get(
                `${this.apiUrl}/wp-json/wp/v2/pages${langParam}${separator}slug=${slug}&_embed`
            );

            // ✅ fallback لو مفيش ترجمة
            if ((!response.data || response.data.length === 0) && lang && lang !== "en") {
                response = await axios.get(
                    `${this.apiUrl}/wp-json/wp/v2/pages?lang=en&slug=${slug}&_embed`
                );
            }

            return response.data[0] || null;
        } catch (error) {
            console.error('Error fetching page:', error);
            return null;
        }
    }

    // 📌 جلب بوستات من نوع معين (custom post type)
    async getPostsByType(
        postType: string,
        lang?: string,
        perPage: number = 10
    ): Promise<Post[]> {
        try {
            const langParam = this.getLanguageParam(lang);
            const separator = langParam ? "&" : "?";
            const response = await axios.get(
                `${this.apiUrl}/wp-json/wp/v2/${postType}${langParam}${separator}per_page=${perPage}&_embed`
            );
            return response.data;
        } catch (error) {
            console.error(`Error fetching ${postType}:`, error);
            return [];
        }
    }

    // 📌 جلب بوست واحد من نوع مخصص عن طريق الـ slug
    async getPostBySlugAndType(
        postType: string,
        slug: string,
        lang?: string
    ): Promise<Post | null> {
        try {
            const langParam = this.getLanguageParam(lang);
            const separator = langParam ? '&' : '?';
            let response = await axios.get(
                `${this.apiUrl}/wp-json/wp/v2/${postType}${langParam}${separator}slug=${slug}&_embed`
            );

            // ✅ fallback لو مفيش ترجمة
            if ((!response.data || response.data.length === 0) && lang && lang !== "en") {
                response = await axios.get(
                    `${this.apiUrl}/wp-json/wp/v2/${postType}?lang=en&slug=${slug}&_embed`
                );
            }

            return response.data[0] || null;
        } catch (error) {
            console.error(`Error fetching ${postType} by slug:`, error);
            return null;
        }
    }

    // 📌 Contact on customizer
    async getContactInfo(): Promise<any> {
        try {
            const res = await fetch(
                `${this.apiUrl}/wp-json/tewido/v1/contact-info`,
                { cache: "no-store" }
            );
            return await res.json();
        } catch (error) {
            console.error("Error fetching contact info:", error);
            return null;
        }
    }

    // 📌 جلب بيانات الفوتر (About EN/AR)
    async getFooter(): Promise<any> {
        try {
            const res = await fetch(
                `${this.apiUrl}/wp-json/tewido/v1/footer`,
                { cache: "no-store" }
            );
            return await res.json();
        } catch (error) {
            console.error("Error fetching footer data:", error);
            return null;
        }
    }

    // 📌 Transalte
    async getTranslations(): Promise<any> {
        try {
            const res = await fetch(`${this.apiUrl}/wp-json/tewido/v1/translations`);
            if (!res.ok) throw new Error("Failed to fetch translations");
            return await res.json();
        } catch (error) {
            console.error("Translations API error:", error);
            return {};
        }
    }




    // 📌 Get all categories
    async getCategories(lang?: string): Promise<any[]> {
        try {
            const langParam = this.getLanguageParam(lang);
            const separator = langParam ? "&" : "?";
            const response = await axios.get(
                `${this.apiUrl}/wp-json/wp/v2/categories${langParam}${separator}hide_empty=true&per_page=100`
            );
            return response.data;
        } catch (error) {
            console.error('Error fetching categories:', error);
            return [];
        }
    }

    // 📌 Get category info by slug
    async getCategoryBySlug(slug: string, lang?: string): Promise<any | null> {
        try {
            const langParam = this.getLanguageParam(lang);
            const separator = langParam ? '&' : '?';
            let response = await axios.get(
                `${this.apiUrl}/wp-json/wp/v2/categories${langParam}${separator}slug=${slug}`
            );

            // ✅ fallback for other language
            if ((!response.data || response.data.length === 0) && lang && lang !== "en") {
                response = await axios.get(
                    `${this.apiUrl}/wp-json/wp/v2/categories?lang=en&slug=${slug}`
                );
            }

            return response.data[0] || null;
        } catch (error) {
            console.error('Error fetching category by slug:', error);
            return null;
        }
    }

    // 📌 Fetch category ID by slug (updated version)
    async getCategoryIdBySlug(slug: string, lang?: string): Promise<number | null> {
        try {
            const langParam = this.getLanguageParam(lang);
            const separator = langParam ? '&' : '?';
            let response = await axios.get(
                `${this.apiUrl}/wp-json/wp/v2/categories${langParam}${separator}slug=${slug}`
            );

            // ✅ fallback for other language
            if ((!response.data || response.data.length === 0) && lang && lang !== "en") {
                response = await axios.get(
                    `${this.apiUrl}/wp-json/wp/v2/categories?lang=en&slug=${slug}`
                );
            }

            if (response.data.length > 0) {
                return response.data[0].id;  // Return category ID
            }
            return null;  // Return null if not found
        } catch (error) {
            console.error("Error fetching category ID by slug:", error);
            return null;
        }
    }

    // 📌 Get posts by category using category slug (updated version)
    async getPostsByCategory(
        slug: string,
        lang?: string,
        perPage: number = 10,
        page: number = 1
    ): Promise<{ items: Post[]; totalPages: number }> {
        // First get category ID
        const categoryId = await this.getCategoryIdBySlug(slug, lang);
        if (!categoryId) {
            return { items: [], totalPages: 1 };
        }

        try {
            const langParam = this.getLanguageParam(lang);
            const separator = langParam ? '&' : '?';

            const response = await axios.get(
                `${this.apiUrl}/wp-json/wp/v2/posts${langParam}${separator}categories=${categoryId}&per_page=${perPage}&page=${page}&_embed`
            );

            return {
                items: response.data,
                totalPages: parseInt(response.headers["x-wp-totalpages"] || "1", 10),
            };
        } catch (error) {
            console.error("Error fetching posts by category:", error);
            return { items: [], totalPages: 1 };
        }
    }







    // 📌 Debug: Get all available taxonomies
    async getAvailableTaxonomies(): Promise<any> {
        try {
            const response = await axios.get(`${this.apiUrl}/wp-json/wp/v2/taxonomies`);
            console.log('Available taxonomies:', response.data);
            return response.data;
        } catch (error) {
            console.error('Error fetching taxonomies:', error);
            return {};
        }
    }

    // 📌 Debug: Get all available post types
    async getAvailablePostTypes(): Promise<any> {
        try {
            const response = await axios.get(`${this.apiUrl}/wp-json/wp/v2/types`);
            console.log('Available post types:', response.data);
            return response.data;
        } catch (error) {
            console.error('Error fetching post types:', error);
            return {};
        }
    }

    // 📌 Get terms from custom taxonomy
    async getCustomTaxonomyTerms(taxonomy: string, lang?: string): Promise<any[]> {
        try {
            const langParam = this.getLanguageParam(lang);
            const separator = langParam ? "&" : "?";

            const response = await axios.get(
                `${this.apiUrl}/wp-json/wp/v2/${taxonomy}${langParam}${separator}hide_empty=true&per_page=100`
            );
            return response.data;
        } catch (error) {
            console.error(`Error fetching ${taxonomy} terms:`, error);
            return [];
        }
    }

    // 📌 Get term by slug from custom taxonomy
    async getCustomTaxonomyTermBySlug(taxonomy: string, slug: string, lang?: string): Promise<any | null> {
        try {
            const langParam = this.getLanguageParam(lang);
            const separator = langParam ? '&' : '?';

            let response = await axios.get(
                `${this.apiUrl}/wp-json/wp/v2/${taxonomy}${langParam}${separator}slug=${slug}`
            );

            // ✅ fallback for other language
            if ((!response.data || response.data.length === 0) && lang && lang !== "en") {
                response = await axios.get(
                    `${this.apiUrl}/wp-json/wp/v2/${taxonomy}?lang=en&slug=${slug}`
                );
            }

            return response.data[0] || null;
        } catch (error) {
            console.error(`Error fetching ${taxonomy} term by slug:`, error);
            return null;
        }
    }

    // 📌 Get posts by custom taxonomy term
    async getPostsByCustomTaxonomy(
        postType: string,
        taxonomy: string,
        termSlug: string,
        lang?: string,
        perPage: number = 10,
        page: number = 1
    ): Promise<{ items: Post[]; totalPages: number }> {
        // First get term ID
        const term = await this.getCustomTaxonomyTermBySlug(taxonomy, termSlug, lang);
        if (!term) {
            return { items: [], totalPages: 1 };
        }

        try {
            const langParam = this.getLanguageParam(lang);
            const separator = langParam ? '&' : '?';

            const response = await axios.get(
                `${this.apiUrl}/wp-json/wp/v2/${postType}${langParam}${separator}${taxonomy}=${term.id}&per_page=${perPage}&page=${page}&_embed`
            );

            return {
                items: response.data,
                totalPages: parseInt(response.headers["x-wp-totalpages"] || "1", 10),
            };
        } catch (error) {
            console.error(`Error fetching ${postType} by ${taxonomy}:`, error);
            return { items: [], totalPages: 1 };
        }
    }


    // 📌 Get taxonomy terms via post meta/embedded data
    async getCustomTaxonomyTermsFromPosts(postType: string, taxonomy: string, lang?: string): Promise<any[]> {
        try {
            const langParam = this.getLanguageParam(lang);
            const separator = langParam ? "&" : "?";

            // Get posts with embedded taxonomy terms
            const response = await axios.get(
                `${this.apiUrl}/wp-json/wp/v2/${postType}${langParam}${separator}per_page=100&_embed`
            );

            // Extract unique taxonomy terms from posts
            const terms = new Map();

            response.data.forEach((post: any) => {
                const postTerms = post._embedded?.["wp:term"] || [];
                postTerms.forEach((termGroup: any[]) => {
                    termGroup.forEach((term: any) => {
                        // Check if this term belongs to our taxonomy (you might need to adjust this check)
                        if (term.taxonomy === taxonomy || term.link?.includes(taxonomy)) {
                            terms.set(term.id, {
                                id: term.id,
                                name: term.name,
                                slug: term.slug,
                                description: term.description || '',
                                count: term.count || 0,
                                taxonomy: term.taxonomy
                            });
                        }
                    });
                });
            });

            return Array.from(terms.values());
        } catch (error) {
            console.error(`Error fetching ${taxonomy} terms from posts:`, error);
            return [];
        }
    }

    // 📌 Get specific term by slug from posts
    async getCustomTaxonomyTermBySlugFromPosts(postType: string, taxonomy: string, slug: string, lang?: string): Promise<any | null> {
        try {
            const terms = await this.getCustomTaxonomyTermsFromPosts(postType, taxonomy, lang);
            return terms.find(term => term.slug === slug) || null;
        } catch (error) {
            console.error(`Error finding ${taxonomy} term by slug:`, error);
            return null;
        }
    }

    // 📌 Get posts by taxonomy term using post filtering
    async getPostsByCustomTaxonomyAlternative(
        postType: string,
        taxonomy: string,
        termSlug: string,
        lang?: string,
        perPage: number = 10,
        page: number = 1
    ): Promise<{ items: Post[]; totalPages: number }> {
        try {
            const langParam = this.getLanguageParam(lang);
            const separator = langParam ? '&' : '?';

            // Get all posts and filter client-side (not ideal for large datasets)
            const response = await axios.get(
                `${this.apiUrl}/wp-json/wp/v2/${postType}${langParam}${separator}per_page=100&_embed`
            );

            // Filter posts that have the specific taxonomy term
            const filteredPosts = response.data.filter((post: any) => {
                const postTerms = post._embedded?.["wp:term"] || [];
                return postTerms.some((termGroup: any[]) =>
                    termGroup.some((term: any) =>
                        term.slug === termSlug && (term.taxonomy === taxonomy || term.link?.includes(taxonomy))
                    )
                );
            });

            // Implement pagination manually
            const startIndex = (page - 1) * perPage;
            const endIndex = startIndex + perPage;
            const paginatedPosts = filteredPosts.slice(startIndex, endIndex);
            const totalPages = Math.ceil(filteredPosts.length / perPage);

            return {
                items: paginatedPosts,
                totalPages: totalPages
            };
        } catch (error) {
            console.error(`Error fetching ${postType} by ${taxonomy}:`, error);
            return { items: [], totalPages: 1 };
        }
    }




}

// 📌 نصدر الكلاس كـ Singleton
export default new WordPressAPI();